/***************************************************************************
*
* Copyright 2010, 2011 BMW Car IT GmbH
* Copyright (C) 2011 DENSO CORPORATION and Robert Bosch Car Multimedia Gmbh
*
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*        http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
****************************************************************************/

#include "WaylandGal2dRenderer.h"
#include "GraphicSystems/Gal2dGraphicSystem.h"
#include "WindowSystems/WaylandGal2dWindowSystem.h"
#include "Configuration.h"
#include "TextureBinders/WaylandGcoSurf.h"

WaylandGal2dRenderer::WaylandGal2dRenderer(ICommandExecutor& executor, Configuration& config)
: BaseRenderer(executor, config)
, m_pWindowSystem(0)
, m_pGraphicSystem(0)
, m_width(0)
, m_height(0)
, m_binder(0)
{
    LOG_DEBUG("WaylandGal2dRenderer", "Created");
}

t_ilm_const_string WaylandGal2dRenderer::pluginGetName() const
{
    return "WaylandGal2dRenderer";
}


bool WaylandGal2dRenderer::start(int width, int height, const char* displayname, int maxIterationDurationInMS)
{
    LOG_DEBUG("WaylandGal2dRenderer",
              "Called, width=" << width << ", "
              "height=" << height << ", "
              "displayname=" << displayname << ", "
              "maxIterationDurationInMS=" << maxIterationDurationInMS);

    gcoHAL gcoHal = NULL;
    gco2D gco2d = NULL;
    struct wl_display* nativeDisplayHandle = NULL;
    m_binder = NULL;
    m_width = width;
    m_height = height;

    // add default screen
    LmScreenList& screenList = m_pScene->getScreenList();
    LmScreen* lmScreen = new LmScreen();
    screenList.push_back(lmScreen);

    // create Wayland windows, register as composite manager etc
    m_pWindowSystem = new WaylandGal2dWindowSystem(mExecutor, displayname, m_width, m_height, m_pScene, m_pInputManager);
    if( m_pWindowSystem == NULL )
    {
        LOG_ERROR("WaylandGal2dRenderer",
                  "Window system is not specified. "
                  "Consider specifying "
                  "WITH_WAYLAND_GAL2D, "
                  "width=" << width << ", "
                  "height=" << height << ", "
                  "displayname=" << displayname << ", "
                  "maxIterationDurationInMS=" << maxIterationDurationInMS);
        goto fail; // TODO bad style
    }
    else
    {
        LOG_DEBUG("WaylandGal2dRenderer",
                  "Created window system, address=" << m_pWindowSystem << ", "
                  "width=" << width << ", "
                  "height=" << height << ", "
                  "displayname=" << displayname << ", "
                  "maxIterationDurationInMS=" << maxIterationDurationInMS);
    }

    m_pGraphicSystem = new Gal2dGraphicSystem(m_width, m_height, &mConfiguration);

    m_pGraphicSystem->setBaseWindowSystem(m_pWindowSystem);

    if (!m_pWindowSystem->init((BaseGraphicSystem<void*, void*>*)m_pGraphicSystem))
    {
        LOG_ERROR("WaylandGal2dRenderer",
                  "Can't initialize graphic system, "
                  "window system address=" << m_pWindowSystem);
        goto fail; // TODO bad style
    }

    if (mConfiguration.getIgnoreEventOnAlpha0())
    {
        m_pInputManager->setGraphicSystem(m_pGraphicSystem);
    }
    // create graphic context from window, init egl etc
    nativeDisplayHandle = m_pWindowSystem->getNativeDisplayHandle();
    gcoHal = m_pGraphicSystem->getGcoHal();
    gco2d = m_pGraphicSystem->getGco2d();

    LOG_DEBUG("WaylandGal2dRenderer",
              "Got nativedisplay handle: " << nativeDisplayHandle
              << " from window system address=" << m_pWindowSystem);

    m_binder = new WaylandGcoSurf(gcoHal, gco2d, nativeDisplayHandle, m_pScene);
    if (m_binder && nativeDisplayHandle)
    {
        m_pGraphicSystem->setTextureBinder(m_binder);

        if (!m_pWindowSystem->start(maxIterationDurationInMS))
        {
            LOG_ERROR("WaylandGal2dRenderer", "Can't start renderer");
            goto fail; // TODO bad style
        }
    }
    else
    {
        goto fail; // TODO bad style
    }
    return true;

    fail: // TODO bad style

    LOG_ERROR("WaylandGal2dRenderer",
              "Initialization failed! "
              "window system address=" << m_pWindowSystem << ", "
              "width=" << width << ", "
              "height=" << height << ", "
              "displayname=" << displayname << ", "
              "maxIterationDurationInMS=" << maxIterationDurationInMS);
    return false;
}

void WaylandGal2dRenderer::stop()
{
    LOG_DEBUG("WaylandGal2dRenderer",
              "Called, window system address=" << m_pWindowSystem);

    m_pWindowSystem->stop();
    if(m_binder)
    {
        delete m_binder;
        m_binder = NULL;
        m_pGraphicSystem->setTextureBinder(m_binder);
    }
    if (m_pGraphicSystem)
    {
        delete m_pGraphicSystem;
        m_pGraphicSystem = NULL;
    }

}

bool WaylandGal2dRenderer::destroyVncbuffer(int screenID,void *p_vncsurfptr)
{
    bool    ret_val = false;

    LOG_DEBUG("WaylandGal2dRenderer",
              "screen_id=" << screenID);

    ret_val =  m_pGraphicSystem->destroyVncbuffer(screenID,p_vncsurfptr) ;
    return ret_val;
}

bool WaylandGal2dRenderer::processVnctouchevent(int screenID,int x,int y,int down)
{
    return(m_pWindowSystem->processVnctouchevent(screenID,x,y,down));
}

bool WaylandGal2dRenderer::processVncmouseevent(int screenID,int x,int y,int buttonmask)
{
    return(m_pWindowSystem->processVncmouseevent(screenID,x,y,buttonmask));
}

bool WaylandGal2dRenderer::processVnckeyboardevent(int screenID,unsigned int keycode, int down)
{
    return(m_pWindowSystem->processVnckeyboardevent(screenID,keycode,down));
}

bool WaylandGal2dRenderer::dumpFramebuffertovncsurf(int screenID) 
{
    bool    ret_val = false;

    LOG_DEBUG("WaylandGal2dRenderer",
              "screen_id=" << screenID);

    ret_val =  m_pGraphicSystem->dumpFramebuffertovncsurf(screenID) ;
    return ret_val;
}

bool WaylandGal2dRenderer::createVncbuffer(int displayID,int bpp,void **pp_vncsurfptr,void **pp_buffaddr)
{
    unsigned int length = 0;
    unsigned int * ScreenIds = NULL;
    bool    ret_val = false;

    LOG_DEBUG("WaylandGal2dRenderer",
              "screen_id=" << displayID<<", "
              "bpp="<<bpp);
    ret_val =  m_pGraphicSystem->createVncbuffer(displayID,bpp,pp_vncsurfptr,pp_buffaddr) ;
    return ret_val;
}

void WaylandGal2dRenderer::doScreenShot(std::string fileToSave, uint screen_id)
{
    LOG_DEBUG("WaylandGal2dRenderer",
              "Called, fileToSave=" << fileToSave << ", "
              "screen_id=" << screen_id << ", "
              "window system address=" << m_pWindowSystem);
    m_pWindowSystem->doScreenShot(fileToSave, screen_id);
}

void WaylandGal2dRenderer::destroyClientBufferShm(Surface *ilmSurf)
{
    LOG_DEBUG("WaylandGal2dRenderer",
              "Called, ilmSurface=" << ilmSurf << ", "
              "window system address=" << m_pWindowSystem);
    m_pWindowSystem->destroyClientBufferShm(ilmSurf);
}

void WaylandGal2dRenderer::doScreenShotOfLayer(std::string fileToSave, uint id)
{
    LOG_DEBUG("WaylandGal2dRenderer",
              "Called, fileToSave=" << fileToSave << ", "
              "layer id=" << id << ", "
              "window system address=" << m_pWindowSystem);
    m_pWindowSystem->doScreenShotOfLayer(fileToSave,id);
}

void WaylandGal2dRenderer::doScreenShotOfSurface(std::string fileToSave, uint id, uint layer_id)
{
    LOG_DEBUG("WaylandGal2dRenderer",
              "Called, fileToSave=" << fileToSave << ", "
              "surface id=" << id << ", "
              "layer id=" << layer_id << ", "
              "window system address=" << m_pWindowSystem);
    m_pWindowSystem->doScreenShotOfSurface(fileToSave,id,layer_id);
}

uint WaylandGal2dRenderer::getNumberOfHardwareLayers(uint screenID)
{
    LOG_DEBUG("WaylandGal2dRenderer",
              "Called, screenID=" << screenID << ", "
              "window system address=" << m_pWindowSystem);
    (void)screenID;

    return 0; // TODO provide real value here
}

uint* WaylandGal2dRenderer::getScreenResolution(uint screenID)
{
    LOG_DEBUG("WaylandGal2dRenderer",
              "Called, screenID=" << screenID << ", "
              "window system address=" << m_pWindowSystem);
    return m_pGraphicSystem->getScreenResolution(screenID);
}

float WaylandGal2dRenderer::getScreenFPS(uint screenID)
{
    LOG_DEBUG("WaylandGal2dRenderer",
              "Called, screenID=" << screenID << ", "
              "window system address=" << m_pWindowSystem);
    return m_pWindowSystem->getScreenFPS(screenID);
}

uint* WaylandGal2dRenderer::getScreenIDs(uint* length)
{
    LOG_DEBUG("WaylandGal2dRenderer",
              "Called, length address=" << length << ", "
              "window system address=" << m_pWindowSystem);
    uint* screenIDS = m_pGraphicSystem->getScreenIDs(length);

    return screenIDS;
}

void WaylandGal2dRenderer::signalWindowSystemRedraw()
{
    LOG_DEBUG("WaylandGal2dRenderer",
              "Called, window system address=" << m_pWindowSystem);
    m_pWindowSystem->signalRedrawEvent();
}

void WaylandGal2dRenderer::forceCompositionWindowSystem()
{
    LOG_DEBUG("WaylandGal2dRenderer",
              "Called, window system address=" << m_pWindowSystem);
    m_pWindowSystem->m_forceComposition = true;
}

bool WaylandGal2dRenderer::setOptimizationMode(OptimizationType id, OptimizationModeType mode)
{
    LOG_DEBUG("WaylandGal2dRenderer",
              "Called, id=" << id << ", mode=" << mode << ", "
              "window system address=" << m_pWindowSystem);
    return m_pGraphicSystem->setOptimizationMode(id, mode);
}

bool WaylandGal2dRenderer::getOptimizationMode(OptimizationType id, OptimizationModeType *mode)
{
    LOG_DEBUG("WaylandGal2dRenderer",
              "Called, id=" << id << ", mode address=" << mode << ", "
              "window system address=" << m_pWindowSystem);
    return m_pGraphicSystem->getOptimizationMode(id, mode);
}

int WaylandGal2dRenderer::getIterationCounter()
{
    LOG_DEBUG("WaylandGal2dRenderer",
              "Called, window system address=" << m_pWindowSystem);
    // TODO: add real thread iteration counter here
    // The renderer plugin thread must wake up at least once in
    // each health monitoring interval. This interval is passed
    // to the plugin as argument in its start() method.
    // Each time the plugin thread gets active, it must
    // increment the internal iteration counter.
    // This way the health monitor can detect, if the plugin
    // thread is running, dead or blocked.

    // TODO: remove this placeholder, which just returns an
    // incremented interation counter to make health monitoring happy.
    static int iteration = 0;
    return ++iteration;
}

bool WaylandGal2dRenderer::isSurfaceDirty(unsigned int surfID)
{
    LOG_DEBUG("WaylandGal2dRenderer",
              "Called, surface id=" << surfID);
	return m_pGraphicSystem->isSurfaceDirty(surfID);
}

bool WaylandGal2dRenderer::isScreenFrozen(unsigned int screenID)
{
    LOG_DEBUG("WaylandGal2dRenderer",
              "Called, screen id=" << screenID);
    return m_pGraphicSystem->isScreenFrozen(screenID);
}

bool WaylandGal2dRenderer::setGamma(unsigned int screenID, double value)
{
    LOG_DEBUG("WaylandGal2dRenderer",
              "Called, screen id=" << screenID <<" ,gamma= " <<value);
    return m_pGraphicSystem->setGamma(screenID,value);
}

bool WaylandGal2dRenderer::setCSC(unsigned int screenID, ilmCSCProperties* pclrprop)
{
    LOG_DEBUG("WaylandGal2dRenderer",
              "Called, screen id=" << screenID);
    return m_pGraphicSystem->setCSC(screenID,pclrprop);
}

InputManager* WaylandGal2dRenderer::getInputManager(char *pSeatName) const
{
    InputManager* pInputManager;
    pInputManager = m_pWindowSystem->getInputManager(pSeatName);
    if (NULL == pInputManager)
    {
        pInputManager = m_pWindowSystem->addInputManager(pSeatName);
    }
    LOG_DEBUG("WaylandGal2dRenderer",
              "Called, seat name=" << pSeatName <<
              " Input Manager=" << pInputManager);
    return pInputManager;
}

bool WaylandGal2dRenderer::getSupportedSeats(ilmInputDevice bitmask,
						t_ilm_string ** pSeatArray, unsigned int *pSizeSeatArray) const
{
    bool ret;
    ret = m_pWindowSystem->getSupportedSeats(bitmask, pSeatArray, pSizeSeatArray);
    return ret;
}

bool WaylandGal2dRenderer::SetDisplayState(unsigned int screenId, ilmScreenState screenState) const
{
    LOG_DEBUG("WaylandGal2dRenderer",
              "Called, screen id=" << screenId <<" ,state= " << screenState);
    return m_pGraphicSystem->setDisplayState(screenId, screenState);
}

bool WaylandGal2dRenderer::GetDisplayState(unsigned int screenId, ilmScreenState *screenState) const
{
    LOG_DEBUG("WaylandGal2dRenderer",
              "Called, screen id=" << screenId);
    return m_pGraphicSystem->getDisplayState(screenId, screenState);
}

bool WaylandGal2dRenderer::SetDisplayAlpha(unsigned int screenId, unsigned int alphaValue) const
{
    LOG_DEBUG("WaylandGal2dRenderer",
              "Called, screen id=" << screenId <<" ,alpha value= " << alphaValue);
    bool ret;
    ret = m_pGraphicSystem->setDisplayAlpha(screenId, alphaValue);
    if(ret)
        m_pWindowSystem->signalRedrawEvent();
    return ret;
}

bool WaylandGal2dRenderer::GetDisplayAlpha(unsigned int screenId, unsigned int *alphaValue) const
{
    LOG_DEBUG("WaylandGal2dRenderer",
              "Called, screen id=" << screenId);
    return m_pGraphicSystem->getDisplayAlpha(screenId, alphaValue);
}
DECLARE_LAYERMANAGEMENT_PLUGIN(WaylandGal2dRenderer)
